#!/bin/sh
# potential_surface_check.sh
# Time-stamp: <2016-08-06 09:39:32 takeshi>
# Author: Takeshi NISHIMATSU
# Ref_1: R. D. King-Smith and David Vanderbilt: PRB 49, 5828 (1994).
# Ref_2: Takeshi Nishimatsu et al.: PRB 82, 134106 (2010).
# Purpose: Basically, you can plot three potential surfaces
#          analytically by using Eqs. (14a)--(14c) in Ref_2.
#          With this test script, You can compare feram data (points)
#          with the analytical plots (lines).
# Results: Resulting potential_surface_check.eps should be
#          similar but not equal to the first-principles
#          potentail surfaces in Fig.2(c) of Ref_2.
##
for alpha in 1 2 3; do
    data_file=potential_surface_$alpha.dat
    rm -f $data_file
    for u in `seq -w 0.00 0.02 0.47`; do
        printf "%s" $u >> $data_file
        uu=`perl -e "print $u/sqrt($alpha)"`
        case "$alpha" in
            1)
                init_dipo_avg="0.0 0.0 $uu" ;;
            2)
                init_dipo_avg="$uu $uu 0.0" ;;
            3)
                init_dipo_avg="$uu $uu $uu" ;;
        esac
        cat > potential_surface_input.feram <<-EOF
	#--- Method, Temperature, and mass ---------------
	method = 'md'
	kelvin = 300
	mass_amu = 38.24
	Q_Nose = 0.1
	
	#--- System geometry -----------------------------
	bulk_or_film = 'bulk'
	L = 16 16 16
	
	#--- Elastic Constants ---------------------------
	B11 = 126.731671475652
	B12 =  41.7582963902598
	B44 =  49.2408864348646
	
	#--- From perovskite-optcell2-p.gp ---------------
	B1xx = -185.347187551195 [eV/Angstrom^2]
	B1yy = -3.28092949275457 [eV/Angstrom^2]
	B4yz = -14.5501738943852 [eV/Angstrom^2]
	P_k1 = -267.98013991724 [eV/Angstrom^6]
	P_k2 = 197.500718362573 [eV/Angstrom^6]
	P_k3 = 830.199979293529 [eV/Angstrom^6]
	P_k4 = 641.968099408642 [eV/Angstrom^8]
	P_alpha = 78.9866142426818 [eV/Angstrom^4]
	P_gamma = -115.484148812672 [eV/Angstrom^4]
	#kappa = -1.51821042113588 [eV/Angstrom^2]
	#(- 8.1460516421 (- (/  -3.812330 2) -1.51821042113588)) =>  8.53400622096412 new P_kappa2
	
	#--- Time step -----------------------------------
	dt = 0.002 [pico second]
	n_thermalize = 1
	n_average    = 1
	n_coord_freq = 3
	coord_directory = 'never'
	distribution_directory = 'never'
	
	#--- From eigenvalues2j --------------------------
	# original  P_kappa2 =    8.1460516421 [eV/Angstrom^2] =    0.0838298622 [Hartree/Bohr^2]
	P_kappa2 = 8.53400622096412
	j =  -2.08403 -1.12904  0.68946 -0.61134  0.00000  0.27690  0.00000    [eV/Angstrom^2]
	a0  =   3.98597    [Angstrom]
	Z_star      =  10.33000
	epsilon_inf =   6.86915
	
	#--- Initial dipole configrations ----------------
	init_dipo_avg = $init_dipo_avg
	init_dipo_dev = 0.00  0.00  0.00   [Angstrom]  # Deviation of initial dipole displacements
EOF
        ./feram potential_surface_input.feram
        grep total_energy potential_surface_input.log | head -1 >> $data_file
    done
done
cat > potential_surface_check.gp <<-EOF
	# potential_surface_check.gp automatically generated by potential_surface_check.sh
	# Do NOT edit this file.
	##
	set encoding iso_8859_1
	set terminal postscript eps enhanced color "Times-Roman" 22
	set output "potential_surface_check.eps"
	set key bottom
	set xrange [0:1]
	set yrange [-0.10:0.15]
	set format y "%.2f"
	set xlabel '{/Times-Italic u} [\305]'
	set ylabel '{/Times-Italic E} - {/Times-Italic E}^0 [eV]'
	call 'potential_surface_input.param.gp'
	HARTREE_BOHR = 14.399644
	kappa = kappa_short - HARTREE_BOHR * 2.0/3.0 * pi * Z_star**2 / epsilon_inf / a0**3
	B = B11  + 2*B12
	C = B1xx + 2*B1yy
	mu_t = (B11  - B12 ) / 2.0
	nu_t = (B1xx - B1yy) / 2.0
	mu_r = B44
	nu_r = B4yz
	alpha_prime = P_alpha - (C**2/B + 4*nu_t**2/mu_t) / 24.0
	gamma_prime = P_gamma + (nu_t**2/mu_t - nu_r**2/mu_r) / 2.0
	E001(x) = kappa * x**2 + alpha_prime * x**4 + P_k1* x**6 + P_k4 * x**8
	E110(x) = kappa * x**2 + (alpha_prime+gamma_prime/4) * x**4 + \\
	          (P_k1+P_k2/4) * x**6           + P_k4 * x**8
	E111(x) = kappa * x**2 + (alpha_prime+gamma_prime/3) * x**4 + \\
	          (P_k1+2*P_k2/9+P_k3/27) * x**6 + P_k4 * x**8
	plot E001(x)      t "{/Times-Italic E}_{001}({/Times-Italic u})" w l lt 1 lw 2,\\
	     'potential_surface_1.dat' u 1:3 t "001 feram data"          w p lt 1,\\
	     E110(x)      t "{/Times-Italic E}_{110}({/Times-Italic u})" w l lt 2 lw 2,\\
	     'potential_surface_2.dat' u 1:3 t "110 feram data"          w p lt 2,\\
	     E111(x)      t "{/Times-Italic E}_{111}({/Times-Italic u})" w l lt 3 lw 2,\\
	     'potential_surface_3.dat' u 1:3 t "111 feram data"          w p lt 3
	set output
EOF
gnuplot  potential_surface_check.gp
#epstopdf potential_surface_check.eps
#Local variables:
#  compile-command: "OMP_NUM_THREADS=2 make check TESTS=potential_surface_check.sh"
#End:
