!==============================================================================
! Module for polynomials (integer coeficients)
!==============================================================================
! Polynoms with integer coefficients are represented by a one-dimensional
! array with variable length (use of ISO TR 15581 extension for the
! ALLOCATABLE-attribut)
!
! Typename: poly
! --------
!  
! example of usage:
! ----------------
!   TYPE(poly) :: p,q,r
!   ...
!   p = poly( (/4, 2, -1/) ! Set p to 4x^2 + 2x - 1
!   q = poly( (/2/) )      ! Set q to 2
!   r = p + q
!   PRINT *, 'Coefficients are: ', r%c
!
! Provided Features:
! -----------------
!    + Operator: Add two polynoms
!    - Operator: Subtract two polynoms
!    * Operator: Multiply two polynoms
!
!------------------------------------------------------------------------------
! Author: Christian G. Eherer
! Compiler: NAGWare Fortran 95 4.0 
!------------------------------------------------------------------------------

MODULE polynom

  TYPE poly
     INTEGER, ALLOCATABLE, DIMENSION(:) :: c
  END TYPE poly

  INTERFACE OPERATOR(+)
     MODULE PROCEDURE addp
  END INTERFACE

  INTERFACE OPERATOR(-)
     MODULE PROCEDURE subp
  END INTERFACE  

  INTERFACE OPERATOR(*)
     MODULE PROCEDURE conv
  END INTERFACE

CONTAINS
 
  !----------------------------------------------------------------------------
  ! add two polynomials
  !----------------------------------------------------------------------------
  FUNCTION addp(a, b)
    USE types
    IMPLICIT NONE
    TYPE(poly), INTENT(in)                     :: a
    TYPE(poly), INTENT(in)                     :: b
    TYPE(poly)                                 :: addp
    ! local variables
    INTEGER                                    :: sa
    INTEGER                                    :: sb

    sa = SIZE(a%c)
    sb = SIZE(b%c)

    ALLOCATE(addp%c(MAX(sa,sb)))

    IF (sa == sb) THEN
       addp%c = a%c + b%c
    ELSE IF (sa < sb) THEN
       addp%c(1:sb-sa) = b%c(1:sb-sa)
       addp%c(sb-sa+1:sb) = a%c(1:sa) + b%c(sb-sa+1:sb)
    ELSE IF (sa > sb) THEN
       addp%c(1:sa-sb) = a%c(1:sa-sb)
       addp%c(sa-sb+1:sa) = b%c(1:sb) + a%c(sa-sb+1:sa)
    END IF

  END FUNCTION addp


  !----------------------------------------------------------------------------
  ! subtract two polynoms, polynoms of order zero are also represented
  ! by a field of length 1, result is reduced 
  !----------------------------------------------------------------------------
  FUNCTION subp(a, b)
    USE types
    IMPLICIT NONE
    TYPE(poly), INTENT(in)                        :: a, b
    TYPE(poly)                                    :: subp
    ! local variables
    TYPE(poly)                                    :: temp
    INTEGER                                       :: bn
   
    bn = SIZE(b%c)
    ALLOCATE(temp%c(bn))
    temp%c = -1 * b%c

    subp = addp(a,temp)

  END FUNCTION subp


  !----------------------------------------------------------------------------
  ! multiply two polynomials
  !----------------------------------------------------------------------------
  ! Convolution, polynomial multiplication, and FIR digital
  ! filtering are all the same operations.
  ! conv(a,b) is the same as conv(b,a), but we can make it go
  ! substantially faster IF we swap arguments to make the first
  ! argument to filter the shorter of the two.
  !----------------------------------------------------------------------------
  FUNCTION conv(a,b)
    USE types
    IMPLICIT NONE
    TYPE(poly), INTENT(in)                        :: a,b
    TYPE(poly)                                    :: conv
    ! local variables
    INTEGER                                       :: na,nb
    TYPE(poly)                                    :: one
    TYPE(poly)                                    :: ta,tb
    
    na = SIZE(a%c)
    nb = SIZE(b%c)
    one = poly( (/1/) )

    ALLOCATE(conv%c(na+nb-1))

    IF (na > nb) THEN
       ALLOCATE(ta%c(na+nb-1))
       ta%c = 0
       ta%c(1:na) = a%c
       conv = filter(b, one, ta)
    ELSE
       ALLOCATE(tb%c(na+nb-1))
       tb%c = 0
       tb%c(1:nb) = b%c
       conv = filter(a, one, tb)
    END IF

  END FUNCTION conv


  !----------------------------------------------------------------------------
  ! function filter One-dimensional digital filter.
  !----------------------------------------------------------------------------
  !   filter(b,a,x) filters the data in vector x with the
  !   filter described by vectors a and b to create the filtered
  !   data.  The filter is a "Direct Form II Transposed"
  !   implementation of the standard difference equation:
  !
  !   a(1)*y(n) = b(1)*x(n) + b(2)*x(n-1) + ... + b(nb+1)*x(n-nb)
  !                         - a(2)*y(n-1) - ... - a(na+1)*y(n-na)
  !
  !   If a(1) is not equal to 1, filter normalizes the filter
  !   coefficients by a(1).
  !----------------------------------------------------------------------------
  FUNCTION filter(b,a,x)
    USE types
    IMPLICIT NONE
    TYPE(poly), INTENT(in)                        :: b,a,x
    TYPE(poly)                                    :: filter
    ! local variables
    INTEGER                                       :: na,nb,nx,cnt,n
    TYPE(poly)                                    :: ta,tb
   
    na = SIZE(a%c)
    nb = SIZE(b%c)
    nx = SIZE(x%c)

    ALLOCATE(filter%c(nx))
    filter%c = 0

    ! usually the following filtering algorithm implies that the
    ! filter coefficients are normalized, so that a%c(1)=1;
    ! for convolution filter is always called in the manner
    ! filter(in,1,vec), so that no normalization is needed to be done
    ! in this case

    IF (na>nb) THEN
       IF (na>=nx) THEN
          ALLOCATE(tb%c(na))
          tb%c = 0
          tb%c(1:nb)=b%c
          ALLOCATE(ta%c(na))
          ta = a;
       ELSE
          ALLOCATE(tb%c(nx))
          tb%c = 0
          tb%c(1:nb)=b%c
          ALLOCATE(ta%c(nx))
          ta%c = 0
          ta%c(1:na)=a%c
       END IF
    ELSEIF (nb>na) THEN
       IF (nb>=nx) THEN
          ALLOCATE(ta%c(nb))
          ta%c = 0
          ta%c(1:na)=a%c
          ALLOCATE(tb%c(nb))
          tb = b
       ELSE
          ALLOCATE(tb%c(nx))
          tb%c = 0
          tb%c(1:nb)=b%c
          ALLOCATE(ta%c(nx))
          ta%c = 0
          ta%c(1:na)=a%c
       END IF
    ELSE
       IF (na<nx) THEN
          ALLOCATE(tb%c(nx))
          tb%c = 0
          tb%c(1:nb)=b%c
          ALLOCATE(ta%c(nx))
          ta%c = 0
          ta%c(1:na)=a%c
       ELSE
          ALLOCATE(ta%c(na))
          ALLOCATE(tb%c(nb))
          ta = a
          tb = b
       END IF
    END IF

    DO n=1,nx
       filter%c(n) = 0
       DO cnt=1,n
          filter%c(n) = filter%c(n) + tb%c(cnt)*x%c(n-cnt+1) &
          - ta%c(cnt)*filter%c(n-cnt+1);
       END DO
    END DO

  END FUNCTION filter


END MODULE polynom









