 
!==============================================================================
! Module for polynomials (rational coefficients)
!==============================================================================
! A rational polynom is represented by two integer lists, with variable length,
! one for the numerator values of the coefficients and one for the 
! denumerators. The ISO TR 15581 extension for the ALLOCATABLE-attribut 
! is used.
! To set up a polynomial a constructor function pol(num, denum) is used. This
! function should always be used to generate polynomials out of integer lists 
! numerator and denonimator. It checks that these lists are of the same length
! and allocate the structure components properly.
! Polynomials of the order zero are also represented by lists of length 1.
! Coefficients that are zero are represented by numerator 0 and denominator 1.
!  
! Typename: ratpoly
! --------
!
! example of usage:
! ----------------
!   TYPE(ratpoly) :: p,q,r,s
!   ...
!   p = pol( (/1, 0, 2/),&  ! Set p to 1/2x^2 + 2/5
!            (/2, 1, 5/) )  
!   q = pol( (/1, 1/),&
!            (/2, 3/) )     ! Set q to 1/2x + 1/3
!   r = p + q
!   s = p * q
!   PRINT *, 'r%n: ', r%n   ! print coefficient lists
!   PRINT *, 'r%d: ', r%d
!   PRINT *, 's%n: ', s%n   
!   PRINT *, 's%d: ', s%d
!
! Provided features:
! -----------------
!    + Operator: Add two polynomials
!    - Operator: Subtract two polynomials
!    * Operator: Multiply two polynomials
!    / Operator: Diviside two polynomials
!    .mod. Operator: Remainder of polynomial division
!
!    In each case the reults are reduced.
!
!------------------------------------------------------------------------------
! Author: Christian G. Eherer
! Compiler: NAGWare Fortran 95 4.0
!----------------------------------------------------------------------------


MODULE rationalpolynom

  TYPE ratpoly
     INTEGER, ALLOCATABLE, DIMENSION(:) :: n
     INTEGER, ALLOCATABLE, DIMENSION(:) :: d
  END TYPE ratpoly

  INTERFACE OPERATOR(+)
     MODULE PROCEDURE addp
  END INTERFACE
  
  INTERFACE OPERATOR(-)
     MODULE PROCEDURE subp
  END INTERFACE

  INTERFACE OPERATOR(*)
     MODULE PROCEDURE conv
  END INTERFACE

  INTERFACE OPERATOR(/)
     MODULE PROCEDURE divide
  END INTERFACE

  INTERFACE OPERATOR(.mod.)
     MODULE PROCEDURE remainder
  END INTERFACE

CONTAINS

!------------------------------------------------------------------------------
! Constructor for rational polynom
!------------------------------------------------------------------------------
! usage: p=pol((/1,2,3/),&
!              (/1,3,4/))
! sets up p as x^2+2/3x+3/4
!------------------------------------------------------------------------------
  FUNCTION pol(num,denum)
    USE types
    IMPLICIT NONE
    TYPE(ratpoly)                                 :: pol
    INTEGER, INTENT(in), DIMENSION(:)             :: num, denum
    ! local variables
    INTEGER                                       :: snum, sdenum

    snum = SIZE(num)
    sdenum = SIZE(denum)

    IF (sdenum/=snum) THEN
       PRINT *,'Error: pol; num and denum have to be of the same length!'
       STOP
    END IF

    ALLOCATE(pol%n(snum))
    ALLOCATE(pol%d(sdenum))

    pol%n = num
    pol%d = denum

  END FUNCTION pol
    


!----------------------------------------------------------------------------
! add two polynoms 
!----------------------------------------------------------------------------
  FUNCTION addp(a, b)
    USE types
    IMPLICIT NONE
    TYPE(ratpoly), INTENT(in)                     :: a, b
    TYPE(ratpoly)                                 :: addp
    ! local variables
    INTEGER                                       :: dn
    INTEGER                                       :: sa
    INTEGER                                       :: sb,i

    sa = SIZE(a%n)
    sb = SIZE(b%d)

    ALLOCATE(addp%n(MAX(sa,sb)))
    ALLOCATE(addp%d(MAX(sa,sb)))

    IF (sa == sb) THEN
       DO i=1,sa
          dn = a%d(i) * b%d(i)
          addp%d(i) = dn
          addp%n(i) = a%n(i) * b%d(i) + b%n(i) * a%d(i)
          IF (addp%n(i)==0) THEN
             addp%d(i) = 1
          END IF
       END DO
    ELSE IF (sa < sb) THEN
       addp%n(1:sb-sa) = b%n(1:sb-sa)
       addp%d(1:sb-sa) = b%d(1:sb-sa)
       DO i=1,sa
         dn = a%d(i) * b%d(sb-sa+i)
         addp%d(sb-sa+i) = dn
         addp%n(sb-sa+i) = a%n(i) * b%d(sb-sa+i) + &
              b%n(sb-sa+i) * a%d(i)
         IF (addp%n(sb-sa+i)==0) THEN
            addp%d(sb-sa+i) = 1
         END IF
      END DO
    ELSE IF (sa > sb) THEN
       addp%n(1:sa-sb) = a%n(1:sa-sb)
       addp%d(1:sa-sb) = a%d(1:sa-sb)
       DO i=1,sb
          dn = b%d(i) * a%d(sa-sb+i)
          addp%d(sa-sb+i) = dn
          addp%n(sa-sb+i) = b%n(i) * a%d(sa-sb+i) + &
               a%n(sa-sb+i) * b%d(i)
          IF (addp%n(sa-sb+i)==0) THEN
             addp%d(sa-sb+i) = 1
          END IF
       END DO
    END IF

    ! reduce coeficients
    addp = reduce(addp)

  END FUNCTION addp

  
!------------------------------------------------------------------------------
! subtract two polynoms
!------------------------------------------------------------------------------
  FUNCTION subp(a, b)
    USE types
    IMPLICIT NONE
    TYPE(ratpoly), INTENT(in)                     :: a, b
    TYPE(ratpoly)                                 :: subp
    ! local variables
    TYPE(ratpoly)                                 :: temp
    INTEGER                                       :: bn
   
    bn = SIZE(b%n)
    ALLOCATE(temp%n(bn))
    ALLOCATE(temp%d(bn))
    temp%n = -1 * b%n
    temp%d = b%d

    subp = addp(a,temp)

  END FUNCTION subp


!------------------------------------------------------------------------------
! multiplication of two rational polynomials, results are reduced; 
!------------------------------------------------------------------------------
  FUNCTION conv(a,b)
    USE types
    IMPLICIT NONE
    TYPE(ratpoly), INTENT(in)                     :: a,b
    TYPE(ratpoly)                                 :: conv
    ! local variables
    INTEGER                                       :: na,nb
    TYPE(ratpoly)                                 :: one
    TYPE(ratpoly)                                 :: ta,tb

    na = SIZE(a%n)
    nb = SIZE(b%n)

    ALLOCATE(conv%n(na+nb-1))
    ALLOCATE(conv%d(na+nb-1))

    ALLOCATE(one%n(1))
    ALLOCATE(one%d(1))
    one%n = 1
    one%d = 1

    ! Convolution, polynomial multiplication, and FIR digital
    ! filtering are all the same operations.
    ! conv(a,b) is the same as conv(b,a), but we can make it go
    ! substantially faster IF we swap arguments to make the first
    ! argument to filter the shorter of the two.
    IF (na > nb) THEN
       ALLOCATE(ta%n(na+nb-1))
       ALLOCATE(ta%d(na+nb-1))
       ta%n = 0
       ta%d = 1
       ta%n(1:na) = a%n
       ta%d(1:na) = a%d
       conv = filter(b, one, ta)
    ELSE
       ALLOCATE(tb%n(na+nb-1))
       ALLOCATE(tb%d(na+nb-1))
       tb%n = 0
       tb%d = 1
       tb%n(1:nb) = b%n
       tb%d(1:nb) = b%d
       conv = filter(a, one, tb)
    END IF

    ! reduce results
    conv = reduce(conv)
    
  END FUNCTION conv
   

!------------------------------------------------------------------------------
! division for rational polynomials, results are reduced, wrapper for deconv
!------------------------------------------------------------------------------
  FUNCTION divide(b,a)
    USE types
    IMPLICIT NONE
    TYPE(ratpoly), INTENT(in)                     :: a,b
    TYPE(ratpoly)                                 :: divide
    ! local variables
    TYPE(ratpoly)                                 :: r

    divide = deconv(b,a,r)

  END FUNCTION divide
    

!------------------------------------------------------------------------------
! remainder of dividing by through a, results are reduced, wrapper for deconv
!------------------------------------------------------------------------------
  FUNCTION remainder(b,a)
    USE types
    IMPLICIT NONE
    TYPE(ratpoly), INTENT(in)                     :: a,b
    TYPE(ratpoly)                                 :: remainder
    ! local variables
    TYPE(ratpoly)                                 :: div

    div = deconv(b,a,remainder)

  END FUNCTION remainder


!------------------------------------------------------------------------------
! deconvolution, division for rational polynomials, results are reduced
!------------------------------------------------------------------------------
  FUNCTION deconv(b,a,r)
    USE types
    IMPLICIT NONE
    TYPE(ratpoly), INTENT(in)                     :: a,b
    TYPE(ratpoly), INTENT(out)                    :: r
    TYPE(ratpoly)                                 :: deconv
    ! local variables
    INTEGER                                       :: na,nb
    TYPE(ratpoly)                                 :: tmp,tmpr

    !deconv Deconvolution and polynomial division.
    !   deconv(b,a) deconvolves vector a out of vector b.  The result
    !   is returned in vector deconv and the remainder in vector r such that
    !   b = conv(a,deconv) + r.
    !
    !   If a and b are vectors of polynomial coefficients, deconvolution
    !   is equivalent to polynomial division (b/a).  
    !   The result of dividing b by a is quotient deconv and remainder r.

    IF (a%n(1)==0) THEN
       PRINT *,'Error: deconv; First coefficient of a must be non-zero.'
       STOP
    END IF

    nb = SIZE(b%n)
    na = SIZE(a%n) 
    IF (na > nb) THEN
       ALLOCATE(deconv%n(1))
       ALLOCATE(deconv%d(1))
       deconv%n=0
       deconv%d=1
       ALLOCATE(r%n(nb))
       ALLOCATE(r%d(nb))
       r = b
       RETURN
    ELSEIF (na==nb) THEN
       ALLOCATE(deconv%n(1))
       ALLOCATE(deconv%d(1))
       ALLOCATE(r%n(nb-1))
       ALLOCATE(r%d(nb-1))
    ELSE ! na < nb
       ALLOCATE(deconv%n(nb-na+1))
       ALLOCATE(deconv%d(nb-na+1))
       ALLOCATE(r%n(na-1))
       ALLOCATE(r%d(na-1))
    END IF
   
    ! Deconvolution and polynomial division are the same operations
    ! as a digital filter's impulse response b(z)/a(z):
    ALLOCATE(tmp%n(nb-na+1))
    ALLOCATE(tmp%d(nb-na+1))
    tmp%n = 0
    tmp%n(1) = 1
    tmp%d = 1
    deconv = filter(b,a,tmp)

    ! calculate remainder;
    tmpr = subp(b,conv(a,deconv)) ! result has the size of b
    ! omit the leading zero coefficients
    r%n = tmpr%n(SIZE(tmpr%n)-SIZE(r%n)+1:SIZE(tmpr%n))
    r%d = tmpr%d(SIZE(tmpr%d)-SIZE(r%d)+1:SIZE(tmpr%d))

    ! reduce results
    deconv = reduce(deconv)
    r = reduce(r)

  END FUNCTION deconv



!------------------------------------------------------------------------------
! function filter One-dimensional digital filter.
!------------------------------------------------------------------------------
!   filter(b,a,x) filters the data in vector x with the
!   filter described by vectors a and a to create the filtered
!   data.  The filter is a "Direct Form II Transposed"
!   implementation of the standard difference equation:
!
!   a(1)*y(n) = b(1)*x(n) + b(2)*x(n-1) + ... + b(nb+1)*x(n-nb)
!                         - a(2)*y(n-1) - ... - a(na+1)*y(n-na)
!
!   If a(1) is not equal to 1, filter normalizes the filter
!   coefficients by a(1). 
!------------------------------------------------------------------------------
  FUNCTION  filter(b,a,x)
    USE types
    IMPLICIT NONE
    TYPE(ratpoly)                                 :: b,a,x
    TYPE(ratpoly)                                 :: filter
    ! local variables
    INTEGER                                       :: na,nb,nx,cnt,n
    TYPE(ratpoly)                                 :: ta,tb
    TYPE(ratpoly)                                 :: an,bn
    TYPE(ratpoly)                                 :: aa,bb,c,d,e,f,g
    INTEGER                                       :: invn, invd
     
    ALLOCATE(filter%n(SIZE(x%n)))
    ALLOCATE(filter%d(SIZE(x%n)))

    filter%n = 0
    filter%d = 1
    na = SIZE(a%n)
    nb = SIZE(b%n)
    nx = SIZE(x%n)
    
    ! copy a,b to be able to reduce and/or normalize them
    ALLOCATE(an%n(na))
    ALLOCATE(an%d(na))
    ALLOCATE(bn%n(nb))
    ALLOCATE(bn%d(nb))
    an = a
    bn = b

    ! normalize, if a(1)/=1
    IF (a%n(1) /= a%d(1)) THEN
       invn = a%d(1)
       invd = a%n(1)
       an%n = a%n*invn
       an%d = a%d*invd
       bn%n = b%n*invn
       bn%d = b%d*invd
    END IF
    an%n(1) = 1
    an%d(1) = 1

    ! reduce an, bn
    an = reduce(an)
    bn = reduce(bn)

    IF (na>nb) THEN
       IF (na>=nx) THEN
          ALLOCATE(tb%n(na))
          ALLOCATE(tb%d(na))
          tb%n = 0
          tb%d = 1
          tb%n(1:nb) = bn%n
          tb%d(1:nb) = bn%d
          ALLOCATE(ta%n(na))
          ALLOCATE(ta%d(na))
          ta = an;
       ELSE
          ALLOCATE(tb%n(nx))
          ALLOCATE(tb%d(nx))
          tb%n = 0
          tb%d = 1
          tb%n(1:nb) = bn%n
          tb%d(1:nb) = bn%d
          ALLOCATE(ta%n(nx))
          ALLOCATE(ta%d(nx))
          ta%n = 0
          ta%d = 1
          ta%n(1:na) = an%n
          ta%d(1:na) = an%d
       END IF
    ELSEIF (nb>na) THEN
       IF (nb>=nx) THEN
          ALLOCATE(ta%n(nb))
          ALLOCATE(ta%d(nb))
          ta%n = 0
          ta%d = 1
          ta%n(1:na) = an%n
          ta%d(1:na) = an%d
          ALLOCATE(tb%n(nb))
          ALLOCATE(tb%d(nb))
          tb = bn
       ELSE
          ALLOCATE(tb%n(nx))
          ALLOCATE(tb%d(nx))
          tb%n = 0
          tb%d = 1
          tb%n(1:nb) = bn%n
          tb%d(1:nb) = bn%d
          ALLOCATE(ta%n(nx))
          ALLOCATE(ta%d(nx))
          ta%n = 0
          ta%d = 1
          ta%n(1:na) = an%n
          ta%d(1:na) = an%d
       END IF
    ELSE
       IF (na<nx) THEN
          ALLOCATE(tb%n(nx))
          ALLOCATE(tb%d(nx))
          tb%n = 0
          tb%d = 1
          tb%n(1:nb) = bn%n
          tb%d(1:nb) = bn%d
          ALLOCATE(ta%n(nx))
          ALLOCATE(ta%d(nx))
          ta%n = 0
          ta%d = 1
          ta%n(1:na) = an%n
          ta%d(1:na) = an%d
       ELSE
          ALLOCATE(ta%n(na))
          ALLOCATE(ta%d(na))
          ALLOCATE(tb%n(nb))
          ALLOCATE(tb%d(nb))
          ta = an
          tb = bn
       END IF
    END IF

    ALLOCATE(aa%n(1))
    ALLOCATE(aa%d(1))
    ALLOCATE(bb%n(1))
    ALLOCATE(bb%d(1))
    ALLOCATE(c%n(1))
    ALLOCATE(c%d(1))
    ALLOCATE(d%n(1))
    ALLOCATE(d%d(1))
    ALLOCATE(e%n(1))
    ALLOCATE(e%d(1))
    ALLOCATE(f%n(1))
    ALLOCATE(f%d(1))
    ALLOCATE(g%n(1))
    ALLOCATE(g%d(1))

    DO n=1,nx
       filter%n(n) = 0
       filter%d(n) = 1
       DO cnt=1,n
          aa%n = filter%n(n)
          aa%d = filter%d(n)
          bb%n = tb%n(cnt)
          bb%d = tb%d(cnt)
          c%n = x%n(n-cnt+1)
          c%d = x%d(n-cnt+1)
          d%n = ta%n(cnt)
          d%d = ta%d(cnt)
          e%n = filter%n(n-cnt+1)
          e%d = filter%d(n-cnt+1)
          f%n = bb%n * c%n
          f%d = bb%d * c%d
          IF (f%n(1)==0) THEN
             f%d(1) = 1
          END IF
          g%n = d%n * e%n
          g%d = d%d * e%d
          IF (g%n(1)==0) THEN
             g%d(1) = 1
          END IF
          filter%d(n) = aa%d(1) * f%d(1) * g%d(1)
          filter%n(n) = aa%n(1)*f%d(1)*g%d(1) + f%n(1)*aa%d(1)*g%d(1) &
               - g%n(1)*aa%d(1)*f%d(1)
          IF (filter%n(n)==0) THEN
             filter%d(n) = 1
          END IF
       END DO
    END DO
 
    ! reduce result
    filter = reduce(filter)

  END FUNCTION filter


!------------------------------------------------------------------------------
! reduces the rational coefficients, using euklid for calculating the gcd
!------------------------------------------------------------------------------
  FUNCTION reduce(p)
    USE types
    IMPLICIT NONE
    TYPE(ratpoly), INTENT(in)                     :: p
    TYPE(ratpoly)                                 :: reduce
    ! local variables
    INTEGER                                       :: i,gd
    
    ALLOCATE(reduce%n(SIZE(p%n)))
    ALLOCATE(reduce%d(SIZE(p%d)))

    DO i=1,SIZE(p%n)
       IF (p%n(i)==0) THEN
          reduce%d(i) = 1
       END IF
       gd = gcd(ABS(p%n(i)),ABS(p%d(i)))
       reduce%n(i) = p%n(i) / gd
       reduce%d(i) = p%d(i) / gd
    END DO

  CONTAINS

    ! ----------------------------------------------------------
    ! Greatest common divisor of two positive integers (Euklid)
    ! added by Christian G. Eherer
    FUNCTION gcd(a,b)
      USE types
      IMPLICIT NONE
      INTEGER, INTENT(in)                           :: a,b
      INTEGER                                       :: gcd
      ! local variables
      INTEGER                                       :: c,aa,bb
      
      aa = a
      bb = b

      IF ((aa==0).OR.(bb==0)) THEN
         gcd = 1
         RETURN
      END IF

      IF (aa < bb) THEN ! since a>=b must be true, they
         c = aa        ! are swapped if a < b
         aa = bb
         bb = c
      END IF
      
      DO
         c = MOD(aa,bb)
         IF (c == 0) EXIT
         aa = bb
         bb = c
      END DO
      
      gcd = bb

    END FUNCTION gcd

  END FUNCTION reduce


END MODULE rationalpolynom










