 
!------------------------------------------------------------------------------
! Module for rational polynomials
! Extended by Christian G. Eherer
!------------------------------------------------------------------------------
MODULE rationalpolynom

  TYPE ratpoly
     INTEGER :: n
     INTEGER :: d
  END TYPE ratpoly

  INTERFACE OPERATOR(+)
     MODULE PROCEDURE addp
  END INTERFACE

  INTERFACE OPERATOR(*)
     MODULE PROCEDURE conv
  END INTERFACE

CONTAINS


  !---------------------------------------------------------
  ! add two polynoms, polynoms of order zero are also represented
  ! by a field of length 1 - extension by C. G. Eherer
  ! result is reduced 
  FUNCTION addp(a, b)
    USE types
    IMPLICIT NONE
    TYPE(ratpoly), INTENT(in),  DIMENSION(:)      :: a, b
    TYPE(ratpoly), DIMENSION(MAX(SIZE(a),SIZE(b))):: addp
    ! local variables
    INTEGER                                       :: dn
    INTEGER                                       :: sa
    INTEGER                                       :: sb,i

    sa = SIZE(a)
    sb = SIZE(b)

    IF (sa == sb) THEN
       DO i=1,sa
          dn = a(i)%d * b(i)%d
          addp(i)%d = dn
          addp(i)%n = a(i)%n * b(i)%d + b(i)%n * a(i)%d
          IF (addp(i)%n==0) THEN
             addp(i)%d = 1
          END IF
       END DO
    ELSE IF (sa < sb) THEN
       addp(1:sb-sa)%n = b(1:sb-sa)%n
       addp(1:sb-sa)%d = b(1:sb-sa)%d
       DO i=1,sa
         dn = a(i)%d * b(sb-sa+i)%d
         addp(sb-sa+i)%d = dn
         addp(sb-sa+i)%n = a(i)%n * b(sb-sa+i)%d + &
              b(sb-sa+i)%n * a(i)%d
         IF (addp(sb-sa+i)%n==0) THEN
            addp(sb-sa+i)%d = 1
         END IF
      END DO
    ELSE IF (sa > sb) THEN
       addp(1:sa-sb)%n = a(1:sa-sb)%n
       addp(1:sa-sb)%d = a(1:sa-sb)%d
       DO i=1,sb
          dn = b(i)%d * a(sa-sb+i)%d
          addp(sa-sb+i)%d = dn
          addp(sa-sb+i)%n = b(i)%n * a(sa-sb+i)%d + &
               a(sa-sb+i)%n * b(i)%d
          IF (addp(sa-sb+i)%n==0) THEN
             addp(sa-sb+i)%d = 1
          END IF
       END DO
    END IF

    ! reduce coeficients
    addp = reduce(addp)

  END FUNCTION addp
  

  !---------------------------------------------------------
  ! this function calculates the proper size for the result of
  ! an addition of two rational polynomials
  FUNCTION addlr(a,b)
    TYPE(ratpoly), INTENT(in), DIMENSION(:)       :: a,b
    INTEGER                                       :: addlr

    addlr = MAX(SIZE(a),SIZE(b))

  END FUNCTION addlr


  !---------------------------------------------------------
  ! multiplication of two rational polynomials, results are
  ! reduced; by Christian G. Eherer
  FUNCTION conv(a,b)
    USE types
    IMPLICIT NONE
    TYPE(ratpoly), INTENT(in), DIMENSION(:)       :: a,b
    TYPE(ratpoly), DIMENSION(SIZE(a)+SIZE(b)-1)   :: conv
    ! local variables
    INTEGER                                       :: na,nb
    TYPE(ratpoly), DIMENSION(1)                   :: one
    TYPE(ratpoly), DIMENSION(:), ALLOCATABLE      :: ta,tb

    na = SIZE(a);
    nb = SIZE(b);
    one%n = 1;
    one%d = 1;

    ! Convolution, polynomial multiplication, and FIR digital
    ! filtering are all the same operations.
    ! conv(a,b) is the same as conv(b,a), but we can make it go
    ! substantially faster IF we swap arguments to make the first
    ! argument to filter the shorter of the two.
    IF (na > nb) THEN
       IF (nb > 1) THEN
          ALLOCATE(ta(na+nb-1))
          ta%n = 0
          ta%d = 1
          ta(1:na) = a
       END IF
       conv = filter(b, one, ta)
    ELSE
       IF (na > 1) THEN
          ALLOCATE(tb(na+nb-1))
          tb%n = 0
          tb%d = 1
          tb(1:nb) = b
       END IF
       conv = filter(a, one, tb)
    END IF

    IF (ALLOCATED(ta)) THEN
       DEALLOCATE(ta)
    END IF

    IF (ALLOCATED(tb)) THEN
       DEALLOCATE(tb)
    END IF

    ! reduce results
    conv = reduce(conv)
    

  END FUNCTION conv
  

  !---------------------------------------------------------
  ! this function gives the length of the result, if you multiply
  ! two polynomials rational a,b
  FUNCTION multrl(a,b)
    USE types
    IMPLICIT NONE
    TYPE(ratpoly), INTENT(in), DIMENSION(:)       :: a,b
    INTEGER                                       :: multrl

    multrl = SIZE(a)+SIZE(b)-1

  END FUNCTION multrl

  !----------------------------------------------------------
  ! deconvolution, division for rational polynomials, results
  ! are reduced
  ! by Christian G. Eherer
  !FUNCTION deconv(b,a,r)
  !  USE types
  !  IMPLICIT NONE
  !  TYPE(ratpoly), INTENT(in),   DIMENSION(:)     :: a,b
  !  TYPE(ratpoly), INTENT(out),  DIMENSION(:)     :: r
  !  TYPE(ratpoly), DIMENSION())      :: deconv
  !  ! local variables
  !  INTEGER                                       :: na,nb
  !  TYPE(ratpoly), ALLOCATABLE,  DIMENSION(:)     :: tmpr

    !deconv Deconvolution and polynomial division.
    !   deconv(b,b) deconvolves vector a out of vector b.  The result
    !   is returned in vector deconv and the remainder in vector r such that
    !   b = conv(a,deconv) + r.
    !
    !   If a and b are vectors of polynomial coefficients, deconvolution
    !   is equivalent to polynomial division.  The result of dividing b by
    !   a is quotient deconv and remainder r.

   ! IF (a(1)%n==0) THEN
   !    PRINT *,'Error: First coefficient of a must be non-zero.'
   !    STOP
   ! END IF

   ! nb = SIZE(b)
   ! na = SIZE(a)
   ! IF (na > nb) THEN
   !    r = b
   !    RETURN
   ! END IF

    ! Deconvolution and polynomial division are the same operations
    ! as a digital filter's impulse response b(z)/a(z):
   ! ALLOCATE(tmp(nb-na+1))
   ! tmp%n = 0
   ! tmp(1)%n = 1
   ! tmp%d = 1
   ! deconv = filter(b,a,tmp)
    
    ! calculate remainder; conv(a,q) has the same length as b, thats ok,
    ! but the left and the right side of an assignment have to be of the
    ! same length; we use a temporary field tmpr
   ! ALLOCATE(tmpr(nb))
   ! tmpr = b - conv(a,q) ! function filter
   ! r = tmpr(SIZE(tmpr):SIZE(tmpr)-SIZE(r)+1)

    ! reduce results
   ! deconv = reduce(deconv)
   ! r = reduce(r)

   ! DEALLOCATE(tmpr)

  !END FUNCTION deconv


  !---------------------------------------------------------
  ! this function returns the right size for the result of the division
  ! b/a in deconvl, and the size of the remainder in r
  !FUNCTION deconvl(b,a,r)
  !  USE types
  !  IMPLICIT NONE
  !  TYPE(ratpoly), INTENT(in), DIMENSION(:)       :: a,b
  !  INTEGER,       INTENT(out)                    :: r
  !  INTEGER                                       :: deconvl
  !  ! local variables
  !  INTEGER                                       :: na,nb

  !  IF (a(1)%n==0) THEN
  !     PRINT *,'Error: First coefficient of a must be non-zero.'
  !     STOP
  !  END IF

  !  nb = SIZE(b)
  !  na = SIZE(a)
  !  ! case, if a is of higher order than b -> no division possible
  !  IF (na > nb) THEN
  !     deconvl = 0
  !     r = SIZE(b)
  !     RETURN
  !  ELSEIF (na==nb) THEN
  !     deconvl = 1
  !     r = nb-1
  !     RETURN
  !  ELSE ! na < nb
  !     deconvl = nb-na+1
  !     r = nb-deconvl
  !     RETURN
  !  END IF

  !END FUNCTION deconvl


  !--------------------------------------------------------
  ! function filter adapted for rational polynomials
  FUNCTION  filter(b,a,x)
    USE types
    IMPLICIT NONE
    TYPE(ratpoly), INTENT(in), DIMENSION(:)       :: b,a,x
    TYPE(ratpoly), DIMENSION(SIZE(x))             :: filter
    ! local variables
    INTEGER                                       :: na,nb,nx,cnt,n
    TYPE(ratpoly), DIMENSION(:), ALLOCATABLE      :: ta,tb
    TYPE(ratpoly), DIMENSION(:), ALLOCATABLE      :: an,bn
    TYPE(ratpoly)                                 :: aa,bb,c,d,e,f,g
    INTEGER                                       :: invn, invd
    
    filter%n = 0
    filter%d = 1
    na = SIZE(a)
    nb = SIZE(b)
    nx = SIZE(x)
    
    ! copy a,b to be able to reduce and/or normalize them
    ALLOCATE(an(na))
    ALLOCATE(bn(nb))
    an = a
    bn = b

    ! normalize, if a(1)/=1
    IF (a(1)%n /= a(1)%d) THEN
       invn = a(1)%d
       invd = a(1)%n
       an%n = a%n*invn
       an%d = a%d*invd
       bn%n = b%n*invn
       bn%d = b%d*invd
    END IF
    an(1)%n = 1
    an(1)%d = 1

    ! reduce an, bn
    an = reduce(an)
    bn = reduce(bn)

    IF (na>nb) THEN
       IF (na>=nx) THEN
          ALLOCATE(tb(na))
          tb%n = 0
          tb%d = 1
          tb(1:nb) = bn
          ALLOCATE(ta(na))
          ta = an;
       ELSE
          ALLOCATE(tb(nx))
          tb%n = 0
          tb%d = 1
          tb(1:nb) = bn
          ALLOCATE(ta(nx))
          ta%n = 0
          ta%d = 1
          ta(1:na) = an
       END IF
    ELSEIF (nb>na) THEN
       IF (nb>=nx) THEN
          ALLOCATE(ta(nb))
          ta%n = 0
          ta%d = 1
          ta(1:na) = an
          ALLOCATE(tb(nb))
          tb = bn
       ELSE
          ALLOCATE(tb(nx))
          tb%n = 0
          tb%d = 1
          tb(1:nb) = bn
          ALLOCATE(ta(nx))
          ta%n = 0
          ta%d = 1
          ta(1:na) = an
       END IF
    ELSE
       IF (na<nx) THEN
          ALLOCATE(tb(nx))
          tb%n = 0
          tb%d = 1
          tb(1:nb) = bn
          ALLOCATE(ta(nx))
          ta%n = 0
          ta%d = 1
          ta(1:na) = an
       ELSE
          ALLOCATE(ta(na))
          ALLOCATE(tb(nb))
          ta = an
          tb = bn
       END IF
    END IF

    DEALLOCATE(an)
    DEALLOCATE(bn)

    DO n=1,nx
       filter(n)%n = 0
       filter(n)%d = 1
       DO cnt=1,n
          aa = filter(n)
          bb = tb(cnt)
          c = x(n-cnt+1)
          d = ta(cnt)
          e = filter(n-cnt+1)
          f%n = bb%n * c%n
          f%d = bb%d * c%d
          IF (f%n==0) THEN
             f%d = 1
          END IF
          g%n = d%n * e%n
          g%d = d%d * e%d
          IF (g%n==0) THEN
             g%d = 1
          END IF
          filter(n)%d = aa%d * f%d * g%d
          filter(n)%n = aa%n*f%d*g%d + f%n*aa%d*g%d - g%n*aa%d*f%d
          IF (filter(n)%n==0) THEN
             filter(n)%d = 1
          END IF
       END DO
    END DO
 
    ! reduce result
    filter = reduce(filter)

    DEALLOCATE(ta)
    DEALLOCATE(tb)

  END FUNCTION filter


  ! ----------------------------------------------------------
  ! reduces the rational coefficients by dividing numerator and 
  ! denominator by their greates common divisor
  FUNCTION reduce(p)
    USE types
    IMPLICIT NONE
    TYPE(ratpoly), INTENT(in), DIMENSION(:)       :: p
    TYPE(ratpoly),             DIMENSION(SIZE(p)) :: reduce
    ! local variables
    INTEGER                                       :: i,gd
    
    DO i=1,SIZE(p)
       IF (p(i)%n==0) THEN
          reduce(i)%d = 1
       END IF
       gd = gcd(ABS(p(i)%n),ABS(p(i)%d))
       reduce(i)%n = p(i)%n / gd
       reduce(i)%d = p(i)%d / gd
    END DO

  CONTAINS

    ! ----------------------------------------------------------
    ! Greatest common divisor of two positive integers (Euklid)
    ! added by Christian G. Eherer
    FUNCTION gcd(a,b)
      USE types
      IMPLICIT NONE
      INTEGER, INTENT(in)                           :: a,b
      INTEGER                                       :: gcd
      ! local variables
      INTEGER                                       :: c,aa,bb
      
      aa = a
      bb = b

      IF ((aa==0).OR.(bb==0)) THEN
         gcd = 1
         RETURN
      END IF

      IF (aa < bb) THEN ! since a>=b must be true, they
         c = aa        ! are swapped if a < b
         aa = bb
         bb = c
      END IF
      
      DO
         c = MOD(aa,bb)
         IF (c == 0) EXIT
         aa = bb
         bb = c
      END DO
      
      gcd = bb

    END FUNCTION gcd

  END FUNCTION reduce


END MODULE rationalpolynom










